<?php
/**
 * Load portability used in the Theme Builder admin page.
 *
 * @since 4.0
 *
 * @return void
 */
function et_theme_builder_load_portability() {
	if ( ! et_pb_is_allowed( 'theme_builder' ) ) {
		return;
	}

	et_core_load_component( 'portability' );
	et_core_portability_register( 'et_theme_builder', array(
		'name' => esc_html__( 'Divi Theme Builder', 'et_builder' ),
		'type' => 'theme_builder',
		'view' => 'et_theme_builder' === et_()->array_get( $_GET, 'page' ),
	) );
}
add_action( 'admin_init', 'et_theme_builder_load_portability' );

/**
 * Register the Theme Builder admin page.
 *
 * @since 4.0
 *
 * @param string $parent
 *
 * @return void
 */
function et_theme_builder_add_admin_page( $parent ) {
	if ( ! et_pb_is_allowed( 'theme_builder' ) ) {
		return;
	}

	// We register the page with the 'read' capability since we check for the ET cap instead.
	add_submenu_page(
		$parent,
		esc_html__( 'Theme Builder', 'et_builder' ),
		esc_html__( 'Theme Builder', 'et_builder' ),
		'read',
		'et_theme_builder',
		'et_theme_builder_admin_page'
	);
}

/**
 * Enqueue Theme Builder assets.
 *
 * @since 4.0
 *
 * @return void
 */
function et_theme_builder_enqueue_scripts() {
	if ( ! et_builder_is_tb_admin_screen() ) {
		return;
	}

	$role_capabilities = et_pb_get_role_settings();
	$user_role         = et_pb_get_current_user_role();

	et_builder_enqueue_open_sans();

	et_fb_enqueue_bundle( 'et-theme-builder', 'theme-builder.css', array( 'et-core-admin' ) );

	et_builder_enqueue_assets_head();
	et_builder_enqueue_assets_main();

	global $wp_version;

	$ver  = ET_BUILDER_VERSION;
	$root = ET_BUILDER_URI;

	if ( version_compare( substr( $wp_version, 0, 3 ), '4.5', '<' ) ) {
		$dep = array( 'jquery-ui-compat' );
		wp_register_script( 'jquery-ui-compat', "{$root}/scripts/ext/jquery-ui-1.10.4.custom.min.js", array( 'jquery' ), $ver, true );
	} else {
		$dep = array( 'jquery-ui-datepicker' );
	}

	wp_register_script( 'jquery-ui-datepicker-addon', "{$root}/scripts/ext/jquery-ui-timepicker-addon.js", $dep, $ver, true );
	wp_register_script( 'react-tiny-mce', "{$root}/frontend-builder/assets/vendors/tinymce.min.js" );

	$asset_id     = 'et-theme-builder';
	$asset_path   = ET_BUILDER_DIR . '/frontend-builder/build/theme-builder.js';
	$asset_uri    = ET_BUILDER_URI . '/frontend-builder/build/theme-builder.js';
	$asset_ver    = ET_BUILDER_VERSION;
	$dependencies = array(
		'jquery',
		'jquery-ui-sortable',
		'jquery-ui-datepicker-addon',
		'react',
		'react-dom',
		'react-tiny-mce',
		'et-core-admin',
		'wp-hooks',
	);

	if ( ! wp_script_is( 'wp-hooks', 'registered' ) ) {
		// Use bundled wp-hooks script when WP < 5.0
		wp_enqueue_script( 'wp-hooks', ET_BUILDER_URI. '/frontend-builder/assets/backports/hooks.js' );
	}

	et_fb_enqueue_react();

	if ( ! file_exists( $asset_path ) ) {
		// Load "hot" from webpack-dev-server.
		$site_url  = wp_parse_url( get_site_url() );
		$asset_uri = "{$site_url['scheme']}://{$site_url['host']}:31495/theme-builder.js";
	}

	wp_enqueue_script( $asset_id, $asset_uri, $dependencies, $asset_ver, true );

	// Strip 'validate' key from settings as it is used server-side only.
	$default_settings   = et_theme_builder_get_template_settings_options();
	foreach ( $default_settings as $group_key => $group ) {
		foreach ( $group['settings'] as $setting_key => $setting ) {
			unset( $default_settings[ $group_key ]['settings'][ $setting_key ]['validate'] );
		}
	}

	$preloaded_settings = et_theme_builder_get_template_settings_options_for_preloading();
	foreach ( $preloaded_settings as $setting_key => $setting ) {
		unset( $preloaded_settings[ $setting_key ]['validate'] );
	}

	$preferences = et_fb_app_preferences();
	$animation   = et_()->array_get( $preferences, 'builder_animation.value', 'true' );
	$animation   = true === $animation || 'true' === $animation;
	$i18n        = require ET_BUILDER_DIR . 'frontend-builder/i18n.php';

	wp_localize_script( 'et-theme-builder', 'et_theme_builder_bundle', array(
		'config' => array(
			'distPath'              => ET_BUILDER_URI . '/frontend-builder/build/',
			'api'                   => admin_url( 'admin-ajax.php' ),
			'apiErrors'             => ET_Theme_Builder_Api_Errors::getMap(),
			'diviLibraryUrl'        => ET_BUILDER_DIVI_LIBRARY_URL,
			'diviLibraryCustomTabs' => apply_filters( 'et_builder_library_modal_custom_tabs', array(), 'theme-builder' ),
			'nonces'                => array(
				'et_builder_library_get_layouts_data'        => wp_create_nonce( 'et_builder_library_get_layouts_data' ),
				'et_theme_builder_api_duplicate_layout'      => wp_create_nonce( 'et_theme_builder_api_duplicate_layout' ),
				'et_theme_builder_api_create_layout'         => wp_create_nonce( 'et_theme_builder_api_create_layout' ),
				'et_theme_builder_api_get_layout_url'        => wp_create_nonce( 'et_theme_builder_api_get_layout_url' ),
				'et_theme_builder_api_save'                  => wp_create_nonce( 'et_theme_builder_api_save' ),
				'et_theme_builder_api_drop_autosave'         => wp_create_nonce( 'et_theme_builder_api_drop_autosave' ),
				'et_theme_builder_api_get_template_settings' => wp_create_nonce( 'et_theme_builder_api_get_template_settings' ),
				'et_theme_builder_api_reset'                 => wp_create_nonce( 'et_theme_builder_api_reset' ),
				'et_theme_builder_api_export_theme_builder'  => wp_create_nonce( 'et_theme_builder_api_export_theme_builder' ),
				'et_theme_builder_api_import_theme_builder'  => wp_create_nonce( 'et_theme_builder_api_import_theme_builder' ),
				'et_builder_library_update_account'          => wp_create_nonce( 'et_builder_library_update_account' ),
			),
			'rtl'              => is_rtl(),
			'animation'        => $animation,
			'templateSettings' => array(
				'default'   => $default_settings,
				'preloaded' => $preloaded_settings,
			),
			'etAccount'        => et_core_get_et_account(),
			'capabilities'     => isset( $role_capabilities[ $user_role ] ) ? $role_capabilities[ $user_role ] : array(),
			'templates' => array(
				'hasDraft' => 0 !== et_theme_builder_get_theme_builder_post_id( false, false ),
				'live'     => et_theme_builder_get_theme_builder_templates( true ),
				'draft'    => et_theme_builder_get_theme_builder_templates( false ),
			),
		),
		'i18n' => array(
			'generic'      => $i18n['generic'],
			'portability'  => $i18n['portability'],
			'library'      => $i18n['library'],
			'themeBuilder' => $i18n['themeBuilder'],
		),
	));
}
add_action( 'admin_enqueue_scripts', 'et_theme_builder_enqueue_scripts' );

/**
 * Render the Theme Builder admin page.
 *
 * @since 4.0
 *
 * @return void
 */
function et_theme_builder_admin_page() {
	echo '<div id="et-theme-builder"></div>';
}
